<?php
namespace App\Http\Controllers\Api;
use App\Events\StockUpdated;
use App\Http\Controllers\Controller;
use App\Models\StockBatch;
use Illuminate\Http\Request;
class StockController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = StockBatch::with(['size', 'subSize', 'brand', 'supplier', 'creator'])
            ->where('status', 'ACTIVE');
        // Search Filter (Batch, Purchase, or Production No)
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('batch_no', 'like', "%{$search}%")
                    ->orWhere('purchase_no', 'like', "%{$search}%");
            });
        }
        // Date Range Filter (Received Date)
        if ($request->has('from_date')) {
            $query->whereDate('received_date', '>=', $request->from_date);
        }
        if ($request->has('to_date')) {
            $query->whereDate('received_date', '<=', $request->to_date);
        }
        // Relational Filters
        if ($request->has('brand_id')) {
            $query->where('brand_id', $request->brand_id);
        }
        if ($request->has('size_id')) {
            $query->where('size_id', $request->size_id);
        }
        if ($request->has('sub_size_id')) {
            $query->where('sub_size_id', $request->sub_size_id);
        }
        if ($request->has('supplier_id')) {
            $query->where('supplier_id', $request->supplier_id);
        }
        // Category Filter (RAW_MATERIAL, FINISHED_GOOD)
        if ($request->has('category')) {
            $query->where('category', $request->category);
        }
        $query->orderBy('id', 'desc');
        $stock = $query->paginate($request->per_page ?? 20);
        return response()->json($stock);
    }
    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $batch = StockBatch::with(['size', 'subSize', 'brand', 'supplier', 'creator'])->findOrFail($id);
        return response()->json($batch);
    }
    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        if ($request->user()->role !== 'ADMIN') {
            return response()->json(['error' => 'Unauthorized. Only admins can add stock.'], 403);
        }
        // Simple validation for MVP
        $data = $request->validate([
            'batch_no' => 'required',
            'purchase_no' => 'required', // Duplicate allowed for multiple items in same bill
            'category' => 'required',
            'size_id' => 'required',
            'total_weight_kg' => 'required|numeric|gt:0',
            'source_type' => 'required',
            'brand_id' => 'nullable|exists:brands,id',
            'received_date' => 'nullable|date',
            'cost_per_kg' => 'nullable|numeric|min:0', // Relaxed from gt:0 as per user request
            'remarks' => 'nullable|string',
            'sub_size_id' => 'nullable',
            'supplier_id' => 'nullable',
        ]);
        // Auto-calculate available fields
        $data['available_weight_kg'] = $data['total_weight_kg'];
        $data['available_pieces'] = $request->input('total_pieces');
        $batch = StockBatch::create(array_merge($request->all(), $data));
        // Broadcast Event
        StockUpdated::dispatch($batch);
        return response()->json($batch, 201);
    }
    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        // Permission Check: Admin can always edit. User can edit only within 3 days.
        $user = $request->user();
        $batch = StockBatch::findOrFail($id);
        if ($batch->is_void) {
            return response()->json(['error' => 'Cannot edit a voided stock batch.'], 403);
        }
        // 3-Day Rule for Non-Admins
        if ($user->role !== 'ADMIN') {
            $diffInDays = $batch->created_at->diffInDays(now());
            if ($diffInDays > 3) {
                return response()->json(['error' => 'You cannot edit this stock after 3 days. Please contact admin.'], 403);
            }
        }
        // Validate strictly for update (similar to store)
        $data = $request->validate([
            'batch_no' => 'required',
            'purchase_no' => 'required',
            'category' => 'required',
            'size_id' => 'required',
            'total_weight_kg' => 'required|numeric|gt:0',
            'source_type' => 'required',
            'brand_id' => 'nullable|exists:brands,id',
            'received_date' => 'nullable|date',
            'cost_per_kg' => 'nullable|numeric|min:0',
            'remarks' => 'nullable|string',
            'total_pieces' => 'nullable|integer',
            'sub_size_id' => 'nullable',
            'supplier_id' => 'nullable',
            'cost_per_piece' => 'nullable|numeric|min:0',
            'bundles_count' => 'nullable|integer',
            'pieces_per_bundle' => 'nullable|integer',
        ]);
        // Recalculate available if total changed? 
        // For MVP, if we edit total_weight, we should adjust available_weight appropriately if no sales happened?
        // Risky if sales happened. 
        // Logic: if sales exist, editing STOCK is dangerous. 
        // Ideally we should block if dependencies exist, but user didn't ask for that.
        // User asked for "Edit within 3 days". 
        // We will update available = (available - old_total + new_total)? 
        // Or just overwrite available if it equals total?
        // Let's assume naive update for now, or just update the master fields.
        // If we change 'total_weight_kg', we need to decide how it affects 'available'.
        // Simplest safe approach: Update fields. If total changed, user must ensure consistency?
        // Let's calculate difference.
        $weightDiff = $data['total_weight_kg'] - $batch->total_weight_kg;
        $piecesDiff = ($request->input('total_pieces') ?? 0) - $batch->total_pieces;
        $batch->fill($data);
        // Adjust available
        $batch->available_weight_kg += $weightDiff;
        $batch->available_pieces += $piecesDiff;
        $batch->save();
        // Broadcast Event
        StockUpdated::dispatch($batch);
        return response()->json($batch);
    }
    public function void(Request $request, string $id)
    {
        // Void can be done by Admin OR User (as per request "even from admin", meaning permanent).
        // But let's restrict to ADMIN or Creator? User said "if any stock is added it can be edited by user within 3 days... once any entry is void it cannot be undone even from admin."
        // Implies User OR Admin can void? Let's allow both for now, or maybe just Admin?
        // "if any stock is added it can be edited... once any entry is void..."
        // I will allow User to void within 3 days too? Or just Admin?
        // Usually VOID is destructive. Let's allow User (within 3 days) and Admin (Always).
        $user = $request->user();
        $batch = StockBatch::findOrFail($id);
        if ($batch->is_void) {
            return response()->json(['error' => 'Stock is already voided.'], 400);
        }
        // 3-Day Rule for Non-Admins to VOID? or can they void anytime?
        // "user can edit within 3 days... once void cannot be undone".
        // I'll apply the same 3-day rule for Voiding to standard users.
        if ($user->role !== 'ADMIN') {
            $diffInDays = $batch->created_at->diffInDays(now());
            if ($diffInDays > 3) {
                return response()->json(['error' => 'You cannot void this stock after 3 days. Please contact admin.'], 403);
            }
        }
        $batch->is_void = true;
        // Optionally set available to 0?
        // Yes, void means it shouldn't be valid stock.
        $batch->available_weight_kg = 0;
        $batch->available_pieces = 0;
        $batch->save();
        StockUpdated::dispatch($batch);
        return response()->json(['message' => 'Stock batch voided successfully', 'batch' => $batch]);
    }
    /**
     * Get the next available batch sequence for a given prefix.
     */
    /**
     * Get the next available batch sequence for a given prefix.
     */
    public function getNextBatchSequence(Request $request)
    {
        $date = $request->input('date') ? \Carbon\Carbon::parse($request->input('date')) : now();
        $column = $request->input('column', 'batch_no');
        $sizeId = $request->input('size_id');
        $subSizeId = $request->input('sub_size_id');
        // Default Logic (if no size provided or for purchase_no)
        if ($column === 'purchase_no' || !$sizeId) {
            // ... existing logic fallback ...
            $fyPrefix = \App\Helpers\SequenceHelper::getFyPrefix($date);
            $staticPrefix = $request->input('static_prefix', '');
            if ($staticPrefix === '') {
                if ($column == 'purchase_no')
                    $staticPrefix = 'PUR';
                elseif ($column == 'batch_no')
                    $staticPrefix = 'R';
            }
            $nextSequence = \App\Helpers\SequenceHelper::generateNextSequence(
                StockBatch::class,
                $column,
                $fyPrefix,
                $staticPrefix
            );
            return response()->json(['sequence' => $nextSequence]);
        }
        // Custom Logic for Batch No (RB/FB)
        // Format: PREFIX-MAIN_SIZE[-SUB_SIZE]-MMDDYYYY-SEQUENCE
        // Determine Prefix (RB for Raw, FB for Ready/Finished)
        // Adjust logic: If called from Raw Stock page, we want RB. If Production, FB.
        // We can pass a 'type' param or infer.
        // For now, if no explicit prefix is sent, default based on something? 
        // Let's assume frontend sends 'static_prefix' = 'RB' or 'FB'.
        $prefix = $request->input('static_prefix', 'RB');
        // Fetch Size Names
        $sizeName = \App\Models\Size::find($sizeId)->name ?? 'X';
        $sizeName = trim(str_ireplace('mm', '', $sizeName));
        $subSizePart = '';
        if ($subSizeId) {
            $subSizeName = \App\Models\SubSize::find($subSizeId)->name ?? '';
            $subSizeName = trim(str_ireplace('mm', '', $subSizeName));
            $subSizeName = str_ireplace('x', '/', $subSizeName); // Replace 'x' or 'X' with '/'
            if ($subSizeName) {
                $subSizePart = '-' . $subSizeName;
            }
        }
        // Date Part: MMDDYYYY
        $datePart = $date->format('mdY');
        // Construct Static Prefix for SequenceHelper
        // We want final: PREFIX-SIZE[-SUBSIZE]-MMDDYYYY-SEQ
        // SequenceHelper does: {static}-{fy}-SEQ
        // SO: static = PREFIX-SIZE[-SUBSIZE]
        // fy = MMDDYYYY
        $constructedStaticPrefix = "{$prefix}-{$sizeName}{$subSizePart}";
        // Sanitize? User example: 6/3. "FB-6-6/3-..."
        // SequenceHelper joins with '-'.
        // SequenceHelper: static . '-' . fy . '-' . seq
        // Result: PREFIX-SIZE[-SUBSIZE]-MMDDYYYY-SEQ
        // This matches exactly if we pass:
        // staticPrefix = "FB-6-6/3" (No trailing dash, helper adds it)
        // fyPrefix = "01212026"
        // Offset for multiple items in same session
        $offset = (int) $request->input('offset', 0);
        $nextSequence = \App\Helpers\SequenceHelper::generateNextSequence(
            StockBatch::class,
            $column,
            $datePart, // Use Date as 'FY'
            $constructedStaticPrefix,
            4,
            $offset
        );
        return response()->json(['sequence' => $nextSequence]);
    }
}