<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

use App\Models\SalesReturn;
use App\Models\SalesOrder;
use App\Models\SalesOrderItem;
use App\Models\StockBatch;
use App\Events\StockUpdated;
use Illuminate\Support\Facades\DB;
use Dompdf\Dompdf;
use Dompdf\Options;


class SalesReturnController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        // Removed 'batch.brand', 'batch.size', 'batch.subSize' as requested
        $query = SalesReturn::with(['order.customer']);

        if ($request->has('from_date') && $request->has('to_date')) {
            $query->whereBetween('created_at', [$request->from_date . ' 00:00:00', $request->to_date . ' 23:59:59']);
        }

        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('invoice_no', 'like', "%{$search}%")
                    ->orWhere('return_no', 'like', "%{$search}%")
                    ->orWhere('batch_no', 'like', "%{$search}%")
                    ->orWhereHas('order.customer', function ($q2) use ($search) {
                        $q2->where('name', 'like', "%{$search}%")
                            ->orWhere('phone', 'like', "%{$search}%");
                    });
            });
        }

        // Relational Filters via Batch
        if ($request->has('brand_id') && $request->brand_id !== 'ALL') {
            $query->whereHas('batch', function ($q) use ($request) {
                $q->where('brand_id', $request->brand_id);
            });
        }
        if ($request->has('size_id') && $request->size_id !== 'ALL') {
            $query->whereHas('batch', function ($q) use ($request) {
                $q->where('size_id', $request->size_id);
            });
        }
        if ($request->has('sub_size_id') && $request->sub_size_id !== 'ALL') {
            $query->whereHas('batch', function ($q) use ($request) {
                $q->where('sub_size_id', $request->sub_size_id);
            });
        }

        return $query->orderBy('id', 'desc')->paginate($request->per_page ?? 10);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        if ($request->user()->role !== 'ADMIN') {
            return response()->json(['error' => 'Unauthorized. Only admins can process returns.'], 403);
        }

        $data = $request->validate([
            'sales_order_id' => 'required|exists:sales_orders,id',
            'items' => 'required|array|min:1',
            'items.*.batch_id' => 'required|exists:stock_batches,id',
            'items.*.quantity_pieces' => 'required|numeric|min:1',
            'items.*.weight_kg' => 'nullable|numeric|min:0',
            'items.*.price_per_piece' => 'required|numeric|min:0',
            'items.*.reason' => 'nullable|string',
        ]);

        try {
            return DB::transaction(function () use ($data) {
                $order = SalesOrder::findOrFail($data['sales_order_id']);

                // Generate Return No: GRN-{YY}{YY+1}-{seq} (e.g. GRN-2526-001)
                // Single GRN for the whole request
                $date = now();
                $year = $date->year;
                $month = $date->month;

                if ($month >= 4) {
                    $fyStart = $year;
                    $fyEnd = $year + 1;
                } else {
                    $fyStart = $year - 1;
                    $fyEnd = $year;
                }
                $shortStart = substr($fyStart, -2);
                $shortEnd = substr($fyEnd, -2);
                $prefix = "GRN-{$shortStart}{$shortEnd}-";

                $latestReturn = SalesReturn::where('return_no', 'like', "{$prefix}%")
                    ->orderBy('id', 'desc')
                    ->first();

                $nextSeq = 1;
                if ($latestReturn) {
                    $parts = explode('-', $latestReturn->return_no);
                    $seqPart = end($parts);
                    $nextSeq = intval($seqPart) + 1;
                }
                $returnNo = $prefix . str_pad($nextSeq, 3, '0', STR_PAD_LEFT);

                $createdReturns = [];
                $totalReturnAmount = 0;

                foreach ($data['items'] as $itemData) {
                    $batch = StockBatch::lockForUpdate()->findOrFail($itemData['batch_id']);

                    // Find item to get ORIGINAL price if needed, but we rely on user input for refund rate
                    $orderItem = SalesOrderItem::where('sales_order_id', $order->id)
                        ->where('batch_id', $batch->id)
                        ->first();

                    if (!$orderItem) {
                        throw new \Exception("Batch {$batch->batch_no} was not part of the selected invoice.");
                    }

                    if ($orderItem->quantity_pieces < $itemData['quantity_pieces']) {
                        throw new \Exception("Cannot return more pieces than purchased for Batch {$batch->batch_no}.");
                    }

                    // Use the custom rate provided by user
                    $pricePerPiece = $itemData['price_per_piece'];
                    $returnAmount = $pricePerPiece * $itemData['quantity_pieces'];
                    $totalReturnAmount += $returnAmount;

                    // Create Return Record
                    $return = SalesReturn::create([
                        'return_no' => $returnNo,
                        'sales_order_id' => $order->id,
                        'invoice_no' => $order->invoice_no,
                        'batch_id' => $batch->id,
                        'batch_no' => $batch->batch_no,
                        'quantity_pieces' => $itemData['quantity_pieces'],
                        'weight_kg' => $itemData['weight_kg'] ?? 0,
                        'price_per_piece' => $pricePerPiece,
                        'amount' => $returnAmount,
                        'reason' => $itemData['reason'] ?? null,
                    ]);

                    $createdReturns[] = $return;

                    // Adjust Stock
                    $batch->available_pieces += $itemData['quantity_pieces'];
                    if (!empty($itemData['weight_kg']) && $itemData['weight_kg'] > 0) {
                        $batch->available_weight_kg += $itemData['weight_kg'];
                    }

                    if ($batch->status === 'CLOSED' && ($batch->available_pieces > 0 || $batch->available_weight_kg > 0)) {
                        $batch->status = 'ACTIVE';
                    }
                    $batch->save();

                    // Update Order Item
                    $orderItem->quantity_pieces -= $itemData['quantity_pieces'];
                    $orderItem->total_weight_kg -= ($itemData['weight_kg'] ?? 0);
                    $orderItem->total_price -= $returnAmount;
                    $orderItem->save();

                    StockUpdated::dispatch($batch);
                }

                // Adjust Order Amount (Revenue) - Once for all items
                $order->total_amount -= $totalReturnAmount;
                // Recalculate tax and net
                $taxAmount = $order->total_amount * ($order->gst_rate / 100);
                $order->tax_amount = $taxAmount;
                $order->net_amount = ($order->total_amount + $taxAmount) - $order->discount_amount;
                $order->save();

                return response()->json($createdReturns, 201);
            });
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 400);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        if (is_numeric($id)) {
            $returnItem = SalesReturn::with(['order.customer', 'batch.brand', 'batch.size', 'batch.subSize'])->findOrFail($id);
        } else {
            // Assume it's a return_no string (e.g. GRN-2526-001)
            // We need to find *one* item to get the group details.
            $returnItem = SalesReturn::with(['order.customer', 'batch.brand', 'batch.size', 'batch.subSize'])
                ->where('return_no', $id)
                ->firstOrFail();
        }

        // Fetch all items belonging to this return number (GRN)
        $allItems = SalesReturn::with(['order.customer', 'batch.brand', 'batch.size', 'batch.subSize'])
            ->where('return_no', $returnItem->return_no)
            ->get();

        return response()->json([
            'return_no' => $returnItem->return_no,
            'created_at' => $returnItem->created_at,
            'invoice_no' => $returnItem->invoice_no,
            'sales_order_id' => $returnItem->sales_order_id,
            'customer' => $returnItem->order->customer,
            'items' => $allItems
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        if ($request->user()->role !== 'ADMIN') {
            return response()->json(['error' => 'Unauthorized. Only admins can update returns.'], 403);
        }

        $data = $request->validate([
            'quantity_pieces' => 'required|numeric|min:0',
            'weight_kg' => 'nullable|numeric|min:0',
            'reason' => 'nullable|string',
        ]);

        try {
            return DB::transaction(function () use ($data, $id) {
                $return = SalesReturn::with('order')->findOrFail($id);

                if ($return->status === 'VOID') {
                    throw new \Exception("Cannot update a VOID return.");
                }

                $batch = StockBatch::lockForUpdate()->findOrFail($return->batch_id);
                $order = SalesOrder::findOrFail($return->sales_order_id);
                $orderItem = SalesOrderItem::where('sales_order_id', $order->id)
                    ->where('batch_id', $batch->id)
                    ->first();

                // 1. Revert Previous Effects
                // Decrease Stock (that was added)
                $batch->available_pieces -= $return->quantity_pieces;
                $batch->available_weight_kg -= $return->weight_kg;

                // Increase Order Item Qty (that was deducted)
                if ($orderItem) {
                    $orderItem->quantity_pieces += $return->quantity_pieces;
                    $orderItem->total_weight_kg += $return->weight_kg;
                    $orderItem->total_price += $return->amount;
                }

                // Increase Order Total (that was refunded)
                $order->total_amount += $return->amount;

                // 2. Validate New Request
                if ($orderItem && ($orderItem->quantity_pieces < $data['quantity_pieces'])) {
                    throw new \Exception("Cannot return more pieces than purchased (Original Qty: {$orderItem->quantity_pieces}).");
                }

                $newReturnAmount = $orderItem ? ($orderItem->price_per_piece * $data['quantity_pieces']) : 0;

                // 3. Apply New Effects
                // Update Return Record
                $return->quantity_pieces = $data['quantity_pieces'];
                $return->weight_kg = $data['weight_kg'] ?? 0;
                $return->amount = $newReturnAmount;
                $return->reason = $data['reason'];
                $return->save();

                // Increase Stock
                $batch->available_pieces += $data['quantity_pieces'];
                if (!empty($data['weight_kg']) && $data['weight_kg'] > 0) {
                    $batch->available_weight_kg += $data['weight_kg'];
                }

                if ($batch->status === 'CLOSED' && ($batch->available_pieces > 0 || $batch->available_weight_kg > 0)) {
                    $batch->status = 'ACTIVE';
                }
                $batch->save();

                // Decrease Order Item Qty
                if ($orderItem) {
                    $orderItem->quantity_pieces -= $data['quantity_pieces'];
                    $orderItem->total_weight_kg -= $data['weight_kg'] ?? 0;
                    $orderItem->total_price -= $newReturnAmount;
                    $orderItem->save();
                }

                // Decrease Order Total (Refund)
                $order->total_amount -= $newReturnAmount;

                // Recalculate Order Tax & Net
                $taxAmount = $order->total_amount * ($order->gst_rate / 100);
                $order->tax_amount = $taxAmount;
                $order->net_amount = ($order->total_amount + $taxAmount) - $order->discount_amount;
                $order->save();

                StockUpdated::dispatch($batch);

                return response()->json($return);
            });
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 400);
        }
    }

    /**
     * Void the specified return.
     */
    public function void(Request $request, $id)
    {
        if ($request->user()->role !== 'ADMIN') {
            return response()->json(['error' => 'Unauthorized. Only admins can void returns.'], 403);
        }

        try {
            return DB::transaction(function () use ($id) {
                $return = SalesReturn::with('order')->findOrFail($id);

                if ($return->status === 'VOID') {
                    return response()->json(['message' => 'Return is already currently voided.']);
                }

                $batch = StockBatch::lockForUpdate()->findOrFail($return->batch_id);
                $order = SalesOrder::findOrFail($return->sales_order_id);
                $orderItem = SalesOrderItem::where('sales_order_id', $order->id)
                    ->where('batch_id', $batch->id)
                    ->first();

                // 1. Revert Stock Addition (Decrease Stock)
                $batch->available_pieces -= $return->quantity_pieces;
                $batch->available_weight_kg -= $return->weight_kg;

                // If stock goes below zero (shouldn't happen logically, but safety check), set to 0 or allow negative?
                // Logic: A return added stock that wasn't actually returned. So we remove it.
                // If it was already sold, available might be low. This might make it negative.
                // Decision: Allow calculation, but user should know. For now standard math.
                $batch->save();

                // 2. Revert Order Financials (Charge back the customer)
                $order->total_amount += $return->amount;

                // Update Order Item if exists (add back qty to sold count)
                if ($orderItem) {
                    $orderItem->quantity_pieces += $return->quantity_pieces;
                    $orderItem->total_weight_kg += $return->weight_kg;
                    $orderItem->total_price += $return->amount;
                    $orderItem->save();
                }

                // Recalculate Order Tax & Net
                $taxAmount = $order->total_amount * ($order->gst_rate / 100);
                $order->tax_amount = $taxAmount;
                $order->net_amount = ($order->total_amount + $taxAmount) - $order->discount_amount;
                $order->save();

                // 3. Mark Return as Void
                $return->status = 'VOID';
                $return->save();

                StockUpdated::dispatch($batch);

                return response()->json($return);
            });
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 400);
        }
    }
    /**
     * Sync (Bulk Update) items for a specific Return Number (GRN).
     * This replaces all existing items for this Return No with the new list.
     */
    public function sync(Request $request, $returnNo)
    {
        if ($request->user()->role !== 'ADMIN') {
            return response()->json(['error' => 'Unauthorized. Only admins can update returns.'], 403);
        }

        $data = $request->validate([
            'sales_order_id' => 'required|exists:sales_orders,id',
            'items' => 'required|array|min:1',
            'items.*.batch_id' => 'required|exists:stock_batches,id',
            'items.*.quantity_pieces' => 'required|numeric|min:1',
            'items.*.weight_kg' => 'nullable|numeric|min:0',
            'items.*.price_per_piece' => 'required|numeric|min:0', // User can edit rate during return edit? Yes.
            'items.*.reason' => 'nullable|string',
        ]);

        try {
            return DB::transaction(function () use ($data, $returnNo) {
                // 1. Fetch Existing Returns for this GRN
                $existingReturns = SalesReturn::where('return_no', $returnNo)->get();

                if ($existingReturns->isEmpty()) {
                    throw new \Exception("Return No {$returnNo} not found.");
                }

                $order = SalesOrder::findOrFail($data['sales_order_id']);

                // 2. Revert effects of ALL existing items (Void Logic without updating status)
                foreach ($existingReturns as $return) {
                    if ($return->status === 'VOID') {
                        continue; // Should not happen if we are editing active returns, but safety.
                    }

                    $batch = StockBatch::lockForUpdate()->find($return->batch_id);
                    $orderItem = SalesOrderItem::where('sales_order_id', $order->id)
                        ->where('batch_id', $return->batch_id)
                        ->first();

                    // Revert Stock (Decrease what was added)
                    if ($batch) {
                        $batch->available_pieces -= $return->quantity_pieces;
                        $batch->available_weight_kg -= $return->weight_kg;
                        $batch->save();
                    }

                    // Revert Financials (Increase Revenue = Charge customer back)
                    $order->total_amount += $return->amount;

                    // Revert Order Item (Add back qty)
                    if ($orderItem) {
                        $orderItem->quantity_pieces += $return->quantity_pieces;
                        $orderItem->total_weight_kg += $return->weight_kg;
                        $orderItem->total_price += $return->amount;
                        $orderItem->save();
                    }

                    // Delete the old row
                    $return->delete();
                }

                // 3. Process NEW items (Store Logic)
                $createdReturns = [];
                $totalNewReturnAmount = 0;

                foreach ($data['items'] as $itemData) {
                    $batch = StockBatch::lockForUpdate()->findOrFail($itemData['batch_id']);

                    $orderItem = SalesOrderItem::where('sales_order_id', $order->id)
                        ->where('batch_id', $batch->id)
                        ->first();

                    if (!$orderItem) {
                        throw new \Exception("Batch {$batch->batch_no} is not part of this invoice.");
                    }

                    // Validation: Ensure we don't return more than what is CURRENTLY in the order item
                    // Note: We just added back the previously returned qty in Step 2, so $orderItem->quantity_pieces is now the full 'pre-return' amount (minus other returns if any).
                    // This logic holds even if user reduces return qty.
                    if ($orderItem->quantity_pieces < $itemData['quantity_pieces']) {
                        throw new \Exception("Cannot return more pieces than purchased (Available: {$orderItem->quantity_pieces}).");
                    }

                    $returnAmount = $itemData['price_per_piece'] * $itemData['quantity_pieces'];
                    $totalNewReturnAmount += $returnAmount;

                    // Create New Return Record
                    $return = SalesReturn::create([
                        'return_no' => $returnNo, // Keep original Return No
                        'sales_order_id' => $order->id,
                        'invoice_no' => $order->invoice_no,
                        'batch_id' => $batch->id,
                        'batch_no' => $batch->batch_no,
                        'quantity_pieces' => $itemData['quantity_pieces'],
                        'weight_kg' => $itemData['weight_kg'] ?? 0,
                        'price_per_piece' => $itemData['price_per_piece'],
                        'amount' => $returnAmount,
                        'reason' => $itemData['reason'] ?? null,
                    ]);

                    $createdReturns[] = $return;

                    // Apply Stock (Increase)
                    $batch->available_pieces += $itemData['quantity_pieces'];
                    if (!empty($itemData['weight_kg']) && $itemData['weight_kg'] > 0) {
                        $batch->available_weight_kg += $itemData['weight_kg'];
                    }
                    if ($batch->status === 'CLOSED') {
                        $batch->status = 'ACTIVE';
                    }
                    $batch->save();

                    // Apply Order Item (Decrease)
                    $orderItem->quantity_pieces -= $itemData['quantity_pieces'];
                    $orderItem->total_weight_kg -= ($itemData['weight_kg'] ?? 0);
                    $orderItem->total_price -= $returnAmount;
                    $orderItem->save();

                    StockUpdated::dispatch($batch);
                }

                // 4. Update Order Financials (Decrease Revenue = Refund)
                $order->total_amount -= $totalNewReturnAmount;

                // Recalculate Tax & Net
                $taxAmount = $order->total_amount * ($order->gst_rate / 100);
                $order->tax_amount = $taxAmount;
                $order->net_amount = ($order->total_amount + $taxAmount) - $order->discount_amount;
                $order->save();

                return response()->json($createdReturns);
            });
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 400);
        }
    }
    public function downloadPdf($id)
    {
        // $id can be return_no or ID.
        if (is_numeric($id)) {
            $returnRecord = SalesReturn::with(['order.customer'])->findOrFail($id);
        } else {
            $returnRecord = SalesReturn::with(['order.customer'])
                ->where('return_no', $id)
                ->firstOrFail();
        }

        $items = SalesReturn::with(['batch.brand', 'batch.size', 'batch.subSize'])
            ->where('return_no', $returnRecord->return_no)
            ->get();

        $customer = $returnRecord->order->customer;

        $options = new Options();
        $options->set('isRemoteEnabled', true);
        $options->set('defaultFont', 'sans-serif');

        $dompdf = new Dompdf($options);

        // Fetch Settings
        $settings = \DB::table('settings')->pluck('value', 'key');

        $totalAmount = $items->sum('amount');

        $html = view('sales_returns.pdf', compact('returnRecord', 'items', 'customer', 'settings', 'totalAmount'))->render();

        $dompdf->loadHtml($html);
        $dompdf->setPaper('A4', 'portrait');
        $dompdf->render();

        return response($dompdf->output())
            ->header('Content-Type', 'application/pdf')
            ->header('Content-Disposition', 'attachment; filename="Credit_Note_' . $returnRecord->return_no . '.pdf"');
    }
}
