<?php

namespace App\Http\Controllers\Api;

use App\Events\StockUpdated;
use App\Http\Controllers\Controller;
use App\Models\ProductionLog;
use App\Models\StockBatch;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ProductionController extends Controller
{
    public function index(Request $request)
    {
        $query = ProductionLog::with(['sourceBatch', 'sourceBatch.size', 'sourceBatch.supplier', 'sourceBatch.brand', 'creator'])
            ->orderBy('created_at', 'desc');

        if ($request->has('from_date') && $request->from_date) {
            $query->whereDate('production_date', '>=', $request->from_date);
        }

        if ($request->has('to_date') && $request->to_date) {
            $query->whereDate('production_date', '<=', $request->to_date);
        }

        // Search by Source Batch No
        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->whereHas('sourceBatch', function ($q) use ($search) {
                $q->where('batch_no', 'like', "%{$search}%");
            });
        }

        if ($request->has('brand_id') && $request->brand_id) {
            $query->whereHas('sourceBatch', function ($q) use ($request) {
                $q->where('brand_id', $request->brand_id);
            });
        }

        if ($request->has('size_id') && $request->size_id) {
            $query->whereHas('sourceBatch', function ($q) use ($request) {
                $q->where('size_id', $request->size_id);
            });
        }

        // Filter by Output Size/SubSize is tricky because no direct relation.
        // We can use a subquery/whereExists on stock_batches
        if ($request->has('output_size_id') && $request->output_size_id) {
            $query->whereExists(function ($q) use ($request) {
                $q->select(DB::raw(1))
                    ->from('stock_batches')
                    ->whereColumn('stock_batches.source_id', 'production_logs.source_batch_id')
                    ->where('stock_batches.source_type', 'PRODUCTION')
                    ->where('stock_batches.size_id', $request->output_size_id)
                    // Use date match strictly
                    ->whereColumn('stock_batches.received_date', 'production_logs.production_date');
            });
        }

        if ($request->has('output_sub_size_id') && $request->output_sub_size_id) {
            $query->whereExists(function ($q) use ($request) {
                $q->select(DB::raw(1))
                    ->from('stock_batches')
                    ->whereColumn('stock_batches.source_id', 'production_logs.source_batch_id')
                    ->where('stock_batches.source_type', 'PRODUCTION')
                    ->where('stock_batches.sub_size_id', $request->output_sub_size_id)
                    ->whereColumn('stock_batches.received_date', 'production_logs.production_date');
            });
        }

        $logs = $query->paginate($request->per_page ?? 20);

        // Attach Output Batches to each Log
        $logs->getCollection()->transform(function ($log) {
            $log->outputs = StockBatch::with(['size', 'subSize'])
                ->where('source_id', $log->source_batch_id)
                ->where('source_type', 'PRODUCTION')
                ->where('received_date', $log->production_date)
                // We use a small window for creation time to be safer, matching the logic in destroy/update
                ->whereBetween('created_at', [$log->created_at->subSeconds(20), $log->created_at->addSeconds(20)])
                ->get();
            return $log;
        });

        return response()->json($logs);
    }

    public function store(Request $request)
    {
        if ($request->user()->role !== 'ADMIN') {
            return response()->json(['error' => 'Unauthorized. Only admins can create production entries.'], 403);
        }

        $data = $request->validate([
            'sourceBatchId' => 'required|exists:stock_batches,id',
            'productionDate' => 'required|date',
            'usedWeight' => 'required|numeric',
            'outputBatches' => 'required|array|min:1',
            'outputBatches.*.batch_no' => 'required|string',
            'outputBatches.*.size_id' => 'required|exists:sizes,id',
            'outputBatches.*.sub_size_id' => 'required|exists:sub_sizes,id',
            'outputBatches.*.bundles_count' => 'required|numeric|min:1',
            'outputBatches.*.pieces_per_bundle' => 'required|numeric|min:1',
            'outputBatches.*.total_weight_kg' => 'required|numeric',
            'outputBatches.*.cost_per_kg' => 'required|numeric',
            'remarks' => 'nullable|string',
        ]);

        try {
            $result = DB::transaction(function () use ($data, $request) {
                // 1. Find Source
                $sourceBatch = StockBatch::lockForUpdate()->find($data['sourceBatchId']);

                // 2. Validate Weight
                if ($sourceBatch->available_weight_kg < $data['usedWeight']) {
                    throw new \Exception("Insufficient weight in source batch. Available: {$sourceBatch->available_weight_kg}kg");
                }

                // 3. Deduct Weight
                $sourceBatch->available_weight_kg -= $data['usedWeight'];
                if ($sourceBatch->available_weight_kg <= 0.1) {
                    $sourceBatch->status = 'CLOSED';
                    $sourceBatch->available_weight_kg = 0;
                }
                $sourceBatch->save();

                // Broadcast Update for Source
                StockUpdated::dispatch($sourceBatch);

                $newBatches = [];
                $totalOutputWeight = 0;

                // 4. Create Output Batches
                foreach ($data['outputBatches'] as $index => $outputData) {
                    $totalOutputWeight += $outputData['total_weight_kg'];

                    // Use provided batch number
                    $batchNo = $outputData['batch_no'];

                    $totalPieces = $outputData['bundles_count'] * $outputData['pieces_per_bundle'];

                    $newBatch = StockBatch::create([
                        'batch_no' => $batchNo,
                        'category' => 'FINISHED_GOOD',
                        'source_type' => 'PRODUCTION',
                        'source_id' => $sourceBatch->id,
                        'brand_id' => $sourceBatch->brand_id, // Inherit brand
                        'received_date' => $data['productionDate'],

                        'size_id' => $outputData['size_id'],
                        'sub_size_id' => $outputData['sub_size_id'],

                        'bundles_count' => $outputData['bundles_count'],
                        'pieces_per_bundle' => $outputData['pieces_per_bundle'],
                        'total_pieces' => $totalPieces,
                        'total_weight_kg' => $outputData['total_weight_kg'],

                        // Derived
                        'weight_per_bundle_kg' => $totalPieces > 0 ? $outputData['total_weight_kg'] / $outputData['bundles_count'] : 0,
                        'weight_per_piece_kg' => $totalPieces > 0 ? $outputData['total_weight_kg'] / $totalPieces : 0,

                        // Available
                        'available_pieces' => $totalPieces,
                        'available_weight_kg' => $outputData['total_weight_kg'],

                        // Costing
                        'cost_per_kg' => $outputData['cost_per_kg'],
                        'cost_per_piece' => $totalPieces > 0 ? ($outputData['cost_per_kg'] * $outputData['total_weight_kg']) / $totalPieces : 0,
                        'remarks' => $request->input('remarks'),
                        'status' => 'ACTIVE'
                    ]);

                    // Broadcast Update for Output
                    StockUpdated::dispatch($newBatch);
                    $newBatches[] = $newBatch;
                }

                // 5. Create Production Log
                ProductionLog::create([
                    'source_batch_id' => $sourceBatch->id,
                    // 'source_initial_weight' => $sourceBatch->available_weight_kg + $data['usedWeight'], // Removed
                    'production_date' => $data['productionDate'],
                    'used_weight' => $data['usedWeight'],
                    'output_total_weight' => $totalOutputWeight,
                    'scrap_weight' => $data['usedWeight'] - $totalOutputWeight,
                    'remarks' => $request->input('remarks'),
                ]);

                return $newBatches;
            });

            return response()->json($result, 201);

        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 400);
        }
    }

    public function destroy(Request $request, $id)
    {
        if ($request->user()->role !== 'ADMIN') {
            return response()->json(['error' => 'Unauthorized. Only admins can delete production entries.'], 403);
        }

        try {
            $result = DB::transaction(function () use ($id) {
                $log = ProductionLog::with('sourceBatch')->find($id);

                if (!$log) {
                    return ['message' => 'Production entry already deleted or not found.'];
                }

                // 1. Find all Output Batches created by this production
                // We know them because their source_id is the sourceBatch->id AND source_type='PRODUCTION' AND created_at matches or via some link? 
                // Ah, ProductionLog doesn't link to Output Batches directly. 
                // But Output Batches link to Source.
                // WE SHOULD HAVE LINKED THEM.
                // However, we can find them by `source_id` = $log->source_batch_id AND `source_type` = 'PRODUCTION' AND `received_date` = $log->production_date AND `created_at` close to log `created_at`.
                // This is risky.
                // Ideally, we should have stored `production_log_id` on StockBatch or `output_batch_ids` on Log.
                // Wait, ProductionLog table doesn't track output batches?
                // Let's check Schema or Migration.

                // Assuming we can find them by: StockBatch::where('source_id', $log->source_batch_id)->where('source_type', 'PRODUCTION')->where('received_date', $log->production_date)->get();
                // But multi-user concurrency might make this slightly ambiguous if two productions happen same day same source.
                // For now, let's use the creation timestamp window or assumption. 
                // Better: Add a column `production_log_id` to `stock_batches` in a migration later? 
                // For now, I will use `created_at` matching within 5 seconds + source_id + source_type + received_date.

                // Let's use a loose match but verify content.
                $outputBatches = StockBatch::where('source_id', $log->source_batch_id)
                    ->where('source_type', 'PRODUCTION')
                    ->where('received_date', $log->production_date)
                    ->whereBetween('created_at', [$log->created_at->subSeconds(5), $log->created_at->addSeconds(5)])
                    ->get();

                if ($outputBatches->count() === 0) {
                    // Fallback: Just look for batches created around that time?
                    // If we can't find them, we can't safely delete.
                    // Actually, if we can't find them, maybe they are already deleted?
                }

                // 2. Check if any output batch has been used/sold
                foreach ($outputBatches as $batch) {
                    // Check if available < total (sold)
                    // Or check if linked to SalesOrderItem (sold)
                    // Or SalesDamage / SalesReturn

                    // Allow small float diff
                    if (abs($batch->available_weight_kg - $batch->total_weight_kg) > 0.01) {
                        throw new \Exception("Cannot delete: Output batch {$batch->batch_no} has already been used or sold.");
                    }

                    // Double check relations
                    // $batch->salesOrderItems()... if relation exists
                }

                // 3. Delete Output Batches
                foreach ($outputBatches as $batch) {
                    $batch->forceDelete(); // Or soft delete? Let's soft delete if model supports it, but here we want to Undo.
                    // StockBatch uses SoftDeletes? No, in my view_file it didn't show SoftDeletes trait usages explicitly in `use` block (Activity, Authors).
                    // Oh, `use ... Auditable;`
                    // Let's assume forceDelete for cleanup or `delete`.
                    $batch->delete();
                }

                // 4. Restore Source Weight
                $source = $log->sourceBatch;
                $source->available_weight_kg += $log->used_weight;

                // Re-open if it was closed
                if ($source->status === 'CLOSED' && $source->available_weight_kg > 0) {
                    $source->status = 'ACTIVE';
                }
                $source->save();

                // 5. Delete Log
                $log->delete();

                StockUpdated::dispatch($source);

                return ['message' => 'Production entry deleted successfully.'];
            });

            return response()->json($result);

        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 400);
        }
    }

    public function show($id)
    {
        $log = ProductionLog::with(['sourceBatch', 'sourceBatch.size', 'sourceBatch.subSize'])->findOrFail($id);
        // We also want to find the output batches associated with this log.
        // Since we don't have a direct relation, we use the heuristic:
        // SourceID + Type=PRODUCTION + Date + CreatedAt window

        $outputBatches = StockBatch::where('source_id', $log->source_batch_id)
            ->where('source_type', 'PRODUCTION')
            ->where('received_date', $log->production_date)
            ->whereBetween('created_at', [$log->created_at->subSeconds(10), $log->created_at->addSeconds(10)])
            ->get();

        $log->outputs = $outputBatches;

        return $log;
    }

    public function update(Request $request, $id)
    {
        if ($request->user()->role !== 'ADMIN') {
            return response()->json(['error' => 'Unauthorized. Only admins can update production entries.'], 403);
        }

        // Validate New Data First
        $data = $request->validate([
            'sourceBatchId' => 'required|exists:stock_batches,id',
            'productionDate' => 'required|date',
            'usedWeight' => 'required|numeric',
            'outputBatches' => 'required|array|min:1',
            'outputBatches.*.batch_no' => 'required|string',
            'outputBatches.*.size_id' => 'required|exists:sizes,id',
            'outputBatches.*.sub_size_id' => 'required|exists:sub_sizes,id',
            'outputBatches.*.bundles_count' => 'required|numeric|min:1',
            'outputBatches.*.pieces_per_bundle' => 'required|numeric|min:1',
            'outputBatches.*.total_weight_kg' => 'required|numeric',
            'outputBatches.*.cost_per_kg' => 'required|numeric',
            'remarks' => 'nullable|string',
        ]);

        try {
            return DB::transaction(function () use ($data, $request, $id) {
                // 1. Revert Old Entry
                $log = ProductionLog::with('sourceBatch')->lockForUpdate()->findOrFail($id);

                // Check Output Batches
                $oldOutputBatches = StockBatch::where('source_id', $log->source_batch_id)
                    ->where('source_type', 'PRODUCTION')
                    ->where('received_date', $log->production_date)
                    ->whereBetween('created_at', [$log->created_at->subSeconds(10), $log->created_at->addSeconds(10)])
                    ->get();

                foreach ($oldOutputBatches as $batch) {
                    if (abs($batch->available_weight_kg - $batch->total_weight_kg) > 0.01) {
                        throw new \Exception("Cannot edit: Old output batch {$batch->batch_no} has already been used or sold.");
                    }
                }

                // Restore Source FROM OLD LOG
                $source = $log->sourceBatch;
                $source->available_weight_kg += $log->used_weight;
                if ($source->status === 'CLOSED' && $source->available_weight_kg > 0) {
                    $source->status = 'ACTIVE';
                }
                $source->save();

                // Delete Old Outputs
                foreach ($oldOutputBatches as $batch) {
                    $batch->delete();
                }

                // Delete Old Log
                $log->delete();

                // 2. Create New Entry

                // Find Source
                $sourceBatch = StockBatch::lockForUpdate()->find($data['sourceBatchId']);

                if ($sourceBatch->available_weight_kg < $data['usedWeight']) {
                    throw new \Exception("Insufficient weight in source batch. Available: {$sourceBatch->available_weight_kg}kg");
                }

                $sourceBatch->available_weight_kg -= $data['usedWeight'];
                if ($sourceBatch->available_weight_kg <= 0.1) {
                    $sourceBatch->status = 'CLOSED';
                    $sourceBatch->available_weight_kg = 0;
                }
                $sourceBatch->save();

                $newBatches = [];
                $totalOutputWeight = 0;

                foreach ($data['outputBatches'] as $outputData) {
                    $totalOutputWeight += $outputData['total_weight_kg'];
                    $totalPieces = $outputData['bundles_count'] * $outputData['pieces_per_bundle'];

                    $newBatch = StockBatch::create([
                        'batch_no' => $outputData['batch_no'],
                        'category' => 'FINISHED_GOOD',
                        'source_type' => 'PRODUCTION',
                        'source_id' => $sourceBatch->id,
                        'brand_id' => $sourceBatch->brand_id,
                        'received_date' => $data['productionDate'],
                        'size_id' => $outputData['size_id'],
                        'sub_size_id' => $outputData['sub_size_id'],
                        'bundles_count' => $outputData['bundles_count'],
                        'pieces_per_bundle' => $outputData['pieces_per_bundle'],
                        'total_pieces' => $totalPieces,
                        'total_weight_kg' => $outputData['total_weight_kg'],
                        'weight_per_bundle_kg' => $totalPieces > 0 ? $outputData['total_weight_kg'] / $outputData['bundles_count'] : 0,
                        'weight_per_piece_kg' => $totalPieces > 0 ? $outputData['total_weight_kg'] / $totalPieces : 0,
                        'available_pieces' => $totalPieces,
                        'available_weight_kg' => $outputData['total_weight_kg'],
                        'cost_per_kg' => $outputData['cost_per_kg'],
                        'cost_per_piece' => $totalPieces > 0 ? ($outputData['cost_per_kg'] * $outputData['total_weight_kg']) / $totalPieces : 0,
                        'remarks' => $request->input('remarks'),
                        'status' => 'ACTIVE'
                    ]);
                    $newBatches[] = $newBatch;
                }

                $newLog = ProductionLog::create([
                    'source_batch_id' => $sourceBatch->id,
                    // 'source_initial_weight' => $sourceBatch->available_weight_kg + $data['usedWeight'], // Removed
                    'production_date' => $data['productionDate'],
                    'used_weight' => $data['usedWeight'],
                    'output_total_weight' => $totalOutputWeight,
                    'scrap_weight' => $data['usedWeight'] - $totalOutputWeight,
                    'remarks' => $request->input('remarks'),
                ]);

                StockUpdated::dispatch($sourceBatch);

                return $newLog;
            });
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 400);
        }
    }
}
