<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Expense;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ExpenseController extends Controller
{
    public function index(Request $request)
    {
        // Group by expense_no (Voucher View)
        $query = Expense::selectRaw('
                expense_no, 
                MAX(expense_date) as expense_date, 
                MAX(created_at) as created_at,
                COUNT(id) as item_count, 
                SUM(amount) as total_amount, 
                MAX(is_void) as is_void,
                MAX(user_id) as user_id 
            ')
            ->groupBy('expense_no')
            ->with('creator');

        // Clone query for total calculation before pagination
        // Note: The main query groups by expense_no. To get the accurate sum of amounts,
        // we should sum the 'amount' column of the matching rows.
        // However, if we group by expense_no, each group is one voucher.
        // We want the sum of total_amount of all vouchers.

        // We can do this by wrapping the grouped query:
        // SELECT SUM(total_amount) FROM (SELECT SUM(amount) as total_amount ... GROUP BY expense_no) as sub

        // OR simpler: just sum 'amount' from the base table matching the filters (ignoring grouping for sum, assuming we want total of all items).
        // Wait, 'amount' is per item. 'total_amount' in group is SUM(amount).
        // So SUM(amount) of all filtered matching rows IS the total expense amount.

        $totalQuery = Expense::query();

        if ($request->has('search') && !empty($request->search)) {
            $search = $request->search;
            $totalQuery->where('expense_no', 'like', "%{$search}%");
        }

        if ($request->has('from_date') && $request->has('to_date')) {
            $totalQuery->whereBetween('expense_date', [$request->from_date, $request->to_date]);
        }

        if ($request->has('category') && $request->category !== 'ALL') {
            $totalQuery->where('category', $request->category);
            // Note: The main query didn't have category filter in the provided file content?
            // Let me double check the file content provided in context.
            // Line 27 only checks 'search'.
            // Line 38 checks dates.
            // I should probably add category filter to both if it's missing in main query too?
            // The prompt implied filter issue.
            // Let's add category support to main query too while I'm here if it's supposed to be there.
            // The frontend passes category.
        }

        $totalFilteredAmount = $totalQuery->sum('amount');

        // Apply same category filter to main query if passed
        if ($request->has('category') && $request->category !== 'ALL') {
            $query->where('category', $request->category);
        }

        // Sorting
        $sortBy = $request->input('sort_by', 'expense_date');
        $sortOrder = $request->input('sort_order', 'desc');

        // Map frontend sort keys to aggregate columns
        $sortMap = [
            'expense_date' => 'MAX(expense_date)', // Order by aggregate
            'amount' => 'SUM(amount)',
            'expense_no' => 'expense_no'
        ];

        $col = $sortMap[$sortBy] ?? 'MAX(expense_date)';
        $query->orderByRaw("$col $sortOrder");

        $perPage = $request->input('per_page', 10);
        $expenses = $query->paginate($perPage);

        return response()->json([
            'data' => $expenses,
            'total_filtered_amount' => $totalFilteredAmount
        ]);
    }

    public function void($expenseNo)
    {
        // Find all items with this expense_no
        $expenses = Expense::where('expense_no', $expenseNo)->get();

        if ($expenses->isEmpty()) {
            return response()->json(['message' => 'Expense voucher not found.'], 404);
        }

        // Check restriction (based on the first item, assume all same date)
        $firstItem = $expenses->first();
        if ($firstItem->created_at->diffInDays(now()) > 3) {
            return response()->json(['message' => 'You cannot void an expense after 3 days of creation. Contact Admin.'], 403);
        }

        // Void all items
        Expense::where('expense_no', $expenseNo)->update(['is_void' => true]);

        return response()->json(['message' => 'Expense voucher voided successfully.']);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'expense_date' => 'required|date',
            'payment_method' => 'required|string',
            'items' => 'required|array|min:1',
            'items.*.category' => 'required|string',
            'items.*.amount' => 'required|numeric|min:0',
            'items.*.description' => 'nullable|string',
            'items.*.reference_no' => 'nullable|string',
        ]);

        $userId = Auth::id();
        $date = \Carbon\Carbon::parse($validated['expense_date']);
        $year = $date->year;
        $month = $date->month;

        if ($month >= 4) {
            // If April or later, e.g., April 2026 -> FY 2026-2027
            $fyStart = $year;
            $fyEnd = $year + 1;
        } else {
            // If before April, e.g., Jan 2026 -> FY 2025-2026
            $fyStart = $year - 1;
            $fyEnd = $year;
        }

        $shortStart = substr($fyStart, -2);
        $shortEnd = substr($fyEnd, -2);
        $fyPrefix = "EXP-{$shortStart}{$shortEnd}-"; // Result: EXP-2526- for Jan 2026

        // Custom sequence generation
        $latestExpense = Expense::where('expense_no', 'like', "{$fyPrefix}%")
            ->orderBy('id', 'desc')
            ->first();

        $nextSeq = 1;
        if ($latestExpense && $latestExpense->expense_no) {
            $parts = explode('-', $latestExpense->expense_no);
            $seqPart = end($parts);
            $nextSeq = intval($seqPart) + 1;
        }

        $expenseNo = $fyPrefix . str_pad($nextSeq, 4, '0', STR_PAD_LEFT);

        $expenses = [];
        foreach ($validated['items'] as $item) {
            $expenses[] = Expense::create([
                'user_id' => $userId,
                'expense_no' => $expenseNo,
                'expense_date' => $validated['expense_date'],
                'payment_method' => $validated['payment_method'],
                'reference_no' => $item['reference_no'] ?? null,
                'category' => $item['category'],
                'amount' => $item['amount'],
                'description' => $item['description'] ?? null,
            ]);
        }

        return response()->json(['message' => 'Expenses created successfully', 'expense_no' => $expenseNo], 201);
    }

    public function update(Request $request, $id)
    {
        $expense = Expense::findOrFail($id);

        // RESTRICTION: Cannot edit after 3 days
        if ($expense->created_at->diffInDays(now()) > 3) {
            return response()->json(['message' => 'You cannot edit an expense after 3 days of adding. Contact Admin.'], 403);
        }

        $validated = $request->validate([
            'expense_date' => 'required|date',
            'category' => 'required|string',
            'amount' => 'required|numeric|min:0',
            'description' => 'nullable|string',
            'payment_method' => 'required|string',
            'reference_no' => 'nullable|string',
        ]);

        $expense->update($validated);

        return response()->json($expense);
    }

    public function destroy($id)
    {
        $expense = Expense::findOrFail($id);

        // RESTRICTION: Cannot delete after 3 days
        if ($expense->created_at->diffInDays(now()) > 3) {
            return response()->json(['message' => 'You cannot delete an expense after 3 days of adding. Contact Admin.'], 403);
        }

        $expense->delete();
        return response()->json(['message' => 'Expense deleted successfully']);
    }
    public function items($expenseNo)
    {
        $expenses = Expense::where('expense_no', $expenseNo)->with('creator')->get();
        if ($expenses->isEmpty()) {
            return response()->json(['message' => 'Expense voucher not found'], 404);
        }
        return response()->json(['data' => $expenses]);
    }

    public function updateBatch(Request $request, $expenseNo)
    {
        // 1. Find existing
        $expenses = Expense::where('expense_no', $expenseNo)->get();
        if ($expenses->isEmpty()) {
            return response()->json(['message' => 'Expense voucher not found.'], 404);
        }

        // 2. Check 3-day restriction on the FIRST item (assuming same date)
        $firstItem = $expenses->first();
        if ($firstItem->created_at->diffInDays(now()) > 3) {
            return response()->json(['message' => 'You cannot edit an expense after 3 days of creation. Contact Admin.'], 403);
        }

        // 3. Validate
        $validated = $request->validate([
            'expense_date' => 'required|date',
            'payment_method' => 'required|string',
            'items' => 'required|array|min:1',
            'items.*.category' => 'required|string',
            'items.*.amount' => 'required|numeric|min:0',
            'items.*.description' => 'nullable|string',
            'items.*.reference_no' => 'nullable|string',
        ]);

        $userId = Auth::id(); // Keep original creator? Or update to editor? Usually keep creator, or track editor. 
        // For simplicity, we re-create, so user_id becomes current user. 
        // If we want to preserve original creator, we should store it.
        $originalUserId = $firstItem->user_id;

        // 4. Delete old items (Hard delete or soft delete? Soft delete is safer but we are replacing)
        // Since we are "updating" the voucher, soft deleting old ones is good history.
        Expense::where('expense_no', $expenseNo)->delete();

        // 5. Create new items with SAME expense_no
        foreach ($validated['items'] as $item) {
            Expense::create([
                'user_id' => $originalUserId, // Preserve creator
                'expense_no' => $expenseNo, // Keep ID
                'expense_date' => $validated['expense_date'],
                'payment_method' => $validated['payment_method'],
                'reference_no' => $item['reference_no'] ?? null,
                'category' => $item['category'],
                'amount' => $item['amount'],
                'description' => $item['description'] ?? null,
            ]);
        }

        return response()->json(['message' => 'Expense voucher updated successfully']);
    }
}
